"""
Generating Videos from Instruments
==================================

This example demonstrates how to use the ``VideoInterface`` to generate
MP4 video files from `arguslib` instruments. The ``VideoInterface`` works
with any instrument that can be rendered directly to an image array, such
as ``DirectCamera``, ``DirectUndistortedCamera``, and interfaces that
wrap them (e.g., ``AircraftInterface``).

The videos generated by this script will be embedded below.
"""

# %%
from arguslib import DirectUndistortedCamera, VideoInterface
from arguslib.aircraft import AircraftInterface
import datetime

cam = DirectUndistortedCamera.from_config("COBALT", "3-7")
dt_start = datetime.datetime(2025, 5, 11, 7, 30)

# When sphinx-gallery runs an example, it executes it in a temporary
# directory. We save the video files directly in this directory (without any
# subfolders) so that the video scraper can find and embed them.

# %%
# Generating a simple camera video
# --------------------------------
#
# Here, we create a ``VideoInterface`` with a ``DirectUndistortedCamera``
# and use ``generate_video`` to create a simple MP4 video from the camera's
# perspective.
video_iface = VideoInterface(cam)
output_video_file = f"cam_video_{dt_start.isoformat(timespec='minutes')}.mp4"

num_frames = 90
video_iface.generate_video(
    output_path=output_video_file,
    start_dt=dt_start,
    end_dt=dt_start + datetime.timedelta(minutes=num_frames - 1),
    step_timedelta=datetime.timedelta(minutes=1),
    fps=4,
    show_kwargs={"brightness_adjust": 1.0},
    time_overlay=True,
)

# %%
# Generating a video with aircraft trails
# ---------------------------------------
#
# The ``VideoInterface`` can wrap other interfaces, like ``AircraftInterface``.
# This allows for creating videos with additional annotations, such as aircraft
# trails.
aci = AircraftInterface(cam)
aci.load_flight_data(dt_start)
video_iface_with_trails = VideoInterface(aci)

output_video_file_trails = f"trails_video_{dt_start.isoformat(timespec='minutes')}.mp4"

num_frames = 90
video_iface_with_trails.generate_video(
    output_path=output_video_file_trails,
    start_dt=dt_start,
    end_dt=dt_start + datetime.timedelta(minutes=num_frames - 1),
    step_timedelta=datetime.timedelta(minutes=1),
    fps=4,
    show_kwargs={
        "brightness_adjust": 1.0,
        "tlen": 15 * 60,
    },
    time_overlay=True,
)
